/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isObject = require( '@stdlib/assert/is-plain-object' );
var hasOwnProp = require( '@stdlib/assert/has-own-property' );
var indexOf = require( './../../index-of' );


// VARIABLES //

var returns = [ 'values', 'keys', '*' ];


// MAIN //

/**
* Validates function options.
*
* @private
* @param {Object} opts - destination object
* @param {Options} options - function options
* @param {*} [options.thisArg] - execution context
* @param {string} [options.returns] - output format
* @returns {(Error|null)} null or an error object
*
* @example
* var opts = {};
* var options = {
*     'returns': '*',
*     'thisArg': {}
* };
* var err = validate( opts, options );
* if ( err ) {
*     throw err;
* }
*/
function validate( opts, options ) {
	if ( !isObject( options ) ) {
		return new TypeError( 'invalid argument. Options must be an object. Value: `' + options + '`.' );
	}
	if ( hasOwnProp( options, 'thisArg' ) ) {
		opts.thisArg = options.thisArg;
	}
	if ( hasOwnProp( options, 'returns' ) ) {
		opts.returns = options.returns;
		if ( indexOf( returns, opts.returns ) === -1 ) {
			return new TypeError( 'invalid option. `returns` option must be one of the following: "' + returns.join( '", "' ) + '". Option: `' + opts.returns + '`.' );
		}
	}
	return null;
}


// EXPORTS //

module.exports = validate;
