/*
 *  @(#)AbstractNewFileServer.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.pmti.v1.autodoc.v1.server;

import java.io.File;
import java.io.IOException;
import java.io.Writer;
import java.io.FileWriter;

import net.sourceforge.groboutils.autodoc.v1.testserver.AbstractWriterServer;
import net.sourceforge.groboutils.autodoc.v1.testserver.TestData;
import net.sourceforge.groboutils.autodoc.v1.testserver.TestInfo;

import org.apache.log4j.Logger;


/**
 * An interface which corresponds to a part of the framework that knows how
 * to deal with the framework's <tt>TestData</tt>.  It may directly deal with
 * the data, or pass it off to a remote server.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since      March 17, 2002
 * @version    $Date: 2003/02/10 22:51:56 $
 */
public abstract class AbstractNewFileServer extends AbstractWriterServer
{
    private static Logger LOG = Logger.getLogger( AbstractNewFileServer.class );
    
    private String prefix;
    private String postfix;
    private boolean useTimestamp = true;
    
    /**
     * Creates a new file for every BugTestResult, in the format
     * <code>prefix + BugTestResult.getSuiteName() + '-' +
     * BugTestResuilt.getTestMethodName() + '-' +
     * Long.toString( System.currentTimeMillis() ) +
     * postfix</code>.  <tt>prefix</tt> should include the directory.
     */
    public AbstractNewFileServer( String prefix, String postfix )
    {
        this( prefix, postfix, true );
    }
    
    /**
     * Creates a new file for every BugTestResult, in the format
     * <code>prefix + BugTestResult.getSuiteName() + '-' +
     * BugTestResuilt.getTestMethodName() + '-' +
     * Long.toString( System.currentTimeMillis() ) +
     * postfix</code>.  <tt>prefix</tt> should include the directory.
     */
    public AbstractNewFileServer( String prefix, String postfix,
            boolean useTimestamp )
    {
        if (prefix == null || postfix == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        this.prefix = prefix;
        this.postfix = postfix;
        this.useTimestamp = useTimestamp;
    }
    
    
    /**
     * Open a writer stream.  This will be done once per result, so log-like
     * actions may need to append to the previous results.
     */
    protected Writer openOutput( TestData td )
            throws IOException
    {
        File f = new File( createFileName( td ) );
        LOG.debug("Opening file "+f.getAbsolutePath()+" for output.");
        FileWriter fw = new FileWriter( f );
        return fw;
    }
    
    
    /**
     * 
     */
    protected String createFileName( TestData td )
    {
        TestInfo ti = td.getTestInfo();
        String suite = "suite";
        String method = "method";
        if (ti != null)
        {
            if (ti.getSuite() != null)
            {
                suite = ti.getSuite();
            }
            if (ti.getMethod() != null)
            {
                method = ti.getMethod();
            }
        }
        StringBuffer sb = new StringBuffer( this.prefix );
        sb.append( suite ).
            append( '-' ).
            append( method );
        if (this.useTimestamp)
        {
            sb.append( '-' ).
                append( System.currentTimeMillis() );
        }
        sb.append( this.postfix );
        String name = sb.toString();
        return name;
    }
}

